# ----------------------------------------------------------------------
# FEniCS 2017.1 code for level set-based structural optimization.
# Written by Antoine Laurain, 2017
# ----------------------------------------------------------------------
from dolfin import *
from init import *
from matplotlib import cm,pyplot as pp
import numpy as np, sys, os
pp.switch_backend('Agg')

# ----------------------------------------------------------------------
def _main():
    Lag,Nx,Ny,lx,ly,Load,Name,ds,bcd,mesh,phi_mat,Vvec=init(sys.argv[1])      
    eps_er, E, nu = [0.01, 20.0, 0.3]  # Elasticity parameters
    mu,lmbda = Constant(E/(2*(1 + nu))),Constant(E*nu/((1+nu)*(1-2*nu)))
    F = project(Expression(("1.0","0.0"),degree=2),Vvec)
    ks = 0.01 # spring parameter    
    # Create folder for saving files
    rd = os.path.join(os.path.dirname(__file__),\
     Name +'/LagVol=' +str(np.int_(Lag))+'_Nx='+str(Nx))
    if not os.path.isdir(rd): os.makedirs(rd) 
    # Line search parameters
    beta0_init,ls,ls_max,gamma,gamma2 = [1.0,0,3,0.8,0.8]   
    beta0 = beta0_init
    beta  = beta0
    # Stopping criterion parameters    
    ItMax,It,stop = [int(2.5*Nx), 0, False]    
    # Cost functional and function space
    J = np.zeros( ItMax )  
    V = FunctionSpace(mesh, 'CG', 1)
    VolUnit = project(Expression('1.0',degree=2),V) # to compute volume 
    U,P = [[0]*len(Load),[0]*len(Load)] # initialize U and P
    # Get vertices coordinates 
    gdim     = mesh.geometry().dim()
    dofsV    = V.tabulate_dof_coordinates().reshape((-1, gdim))    
    dofsVvec = Vvec.tabulate_dof_coordinates().reshape((-1, gdim))     
    px,py    = [(dofsV[:,0]/lx)*2*Nx, (dofsV[:,1]/ly)*2*Ny]
    pxvec,pyvec = [(dofsVvec[:,0]/lx)*2*Nx, (dofsVvec[:,1]/ly)*2*Ny]  
    dofsV_max, dofsVvec_max =((Nx+1)*(Ny+1) + Nx*Ny)*np.array([1,2])    
    # Initialize phi  
    phi = Function( V )  
    phi = _comp_lsf(px,py,phi,phi_mat,dofsV_max) 
    # Define Omega = {phi<0}     
    class Omega(SubDomain):
        def inside(self, x, on_boundary):
            return .0 <= x[0] <= lx and .0 <= x[1] <= ly and phi(x) < 0   
    class Fixed(SubDomain):
        def inside(self, x, on_boundary):
            return (between(x[0],(.0,.05)) and between(x[1],(.48,.52)))\
             or  (between(x[0],(.9,1.0)) and between(x[1],(.43,.57)))
    domains = MeshFunction("size_t", mesh, mesh.topology().dim())
    n  = FacetNormal(mesh) 
    fixed = Fixed()
    domains.set_all(0)       
    fixed.mark(domains, 2) 
    dx = Measure('dx')(subdomain_data = domains)           
    # Define solver to compute descent direction th
    theta,xi = [TrialFunction(Vvec), TestFunction( Vvec)]     
    av = assemble((inner(grad(theta),grad(xi)) +0.1*inner(theta,xi))*dx(0)\
         +1.0e5*inner(theta,xi) * dx(2)\
         +1.0e5*(inner(dot(theta,n),dot(xi,n)) * (ds(0)+ds(1)+ds(2)+ds(3))) ) 
    solverav = LUSolver(av)

    #---------- MAIN LOOP ----------------------------------------------
    while It < ItMax and stop == False:
        # Update and tag Omega = {phi<0}, then solve elasticity system.  
        omega = Omega()
        domains.set_all(0)
        omega.mark(domains, 1)
        fixed.mark(domains, 2) 
        dx = Measure('dx')(subdomain_data = domains)   
        for k in range(0,len(Load)):   
            U[k] = _solve_pde(Vvec,dx,ds,eps_er,bcd,mu,lmbda,Load[k],ks)  
            P[k] = _solve_adj(Vvec,dx,ds,eps_er,bcd,mu,lmbda,F,U[k],ks)    
        fileu   = File("ux.pvd") 
        Ux, Uy = U[0].split()
        Ux = project(Ux,V)    
        fileu << Ux                  
                         
        # Update cost functional 
        objective = 0
        for u in U:
            S1 = inner(F,u)
            objective += assemble(S1*ds(2) + 2.0*S1*ds(3))
        vol = assemble(VolUnit*dx(1) + VolUnit*dx(2))             
        J[It]  = objective + Lag * vol                           
        # ------- LINE SEARCH ------------------------------------------
        if It > 0 and J[It] > J[It-1] and ls < ls_max:
            ls   += 1
            beta *= gamma            
            phi_mat,phi = [phi_mat_old,phi_old]
            phi_mat = _hj(th_mat, phi_mat, lx,ly,Nx, Ny, beta)
            phi     = _comp_lsf(px,py,phi,phi_mat,dofsV_max) 
            print('Line search iteration : %s' % ls)           
        else:          
            print('************ ITERATION NUMBER %s' % It)                   
            print('Function value        : %.5f' % J[It])
            print('Objective             : %.5f' % objective)
            print('Volume fraction       : %.2f' % (vol/(lx*ly))) 
            # Decrease or increase line search step
            if ls == ls_max: beta0 = max(beta0 * gamma2, 0.1*beta0_init)  
            if ls == 0:      beta0 = min(beta0 / gamma2, 1)
            # Reset beta and line search index    
            ls,beta,It = [0,beta0, It+1]
            # Compute the descent direction th           
            th = _shape_der(Vvec,U,P,eps_er,mu,lmbda,dx,solverav, Lag)
            th_array = th.vector().get_local()
            th_mat = [np.zeros((Ny+1,Nx+1)),np.zeros((Ny+1,Nx+1))]          
            for dof in range(0, dofsVvec_max,2):
                if np.rint(pxvec[dof]) %2 == .0:
                    cx,cy= np.int_(np.rint([pxvec[dof]/2,pyvec[dof]/2]))
                    th_mat[0][cy,cx] = th_array[dof]
                    th_mat[1][cy,cx] = th_array[dof+1]                           
            # Update level set function phi using descent direction th
            phi_old, phi_mat_old = [phi, phi_mat]            
            phi_mat = _hj(th_mat, phi_mat, lx,ly,Nx,Ny, beta)
            if np.mod(It,5) == 0: phi_mat = _reinit(lx,ly,Nx,Ny,phi_mat)     
            phi = _comp_lsf(px,py,phi,phi_mat,dofsV_max)                      
            #------------ STOPPING CRITERION ---------------------------
            if It>20 and max(abs(J[It-5:It]-J[It-1]))<2.0*J[It-1]/Nx**2: 
                stop = True  
            #------------ Plot Geometry --------------------------------  
            if np.mod(It,3)==0 or It==1 or It==ItMax or stop==True:   
                pp.close(); ax = pp.subplot()
                ax.contourf(phi_mat,[-10.0,.0],extent = [.0,lx,.0,ly],\
                 cmap=cm.get_cmap('bone'))
                ax.set_aspect('equal','box')
                pp.show()
                pp.savefig(rd+'/it_'+str(It)+'.pdf',bbox_inches='tight')
    return
# ----------------------------------------------------------------------
def _solve_pde(V, dx, ds, eps_er, bcd, mu, lmbda, Load,ks):
    u,v = [TrialFunction(V), TestFunction(V)]
    S1 = 2.0*mu*inner(sym(grad(u)),sym(grad(v))) + lmbda*div(u)*div(v)   
    A = assemble( eps_er*S1*dx(0) +S1*dx(1) +S1*dx(2) +ks*inner(u,v)*ds(2))   
    b = assemble( inner(Expression(('0.0', '0.0'),degree=2) ,v) * ds(2))    
    U = Function(V)
    delta = PointSource(V.sub(0), Load, 0.05)
    delta.apply(b) 
    for bc in bcd: bc.apply(A,b)    
    solver = LUSolver(A)
    solver.solve(U.vector(), b)    
    return U
# ----------------------------------------------------------------------
def _solve_adj(V, dx, ds, eps_er, bcd, mu, lmbda, F, u, ks):
    p,v = [TrialFunction(V), TestFunction(V)]
    S1 = 2.0*mu*inner(sym(grad(p)),sym(grad(v))) + lmbda*div(p)*div(v)
    A = assemble( eps_er*S1*dx(0) +S1*dx(1) +S1*dx(2)+ks*inner(p,v)*ds(2))   
    f_adj = -inner(F,v)       
    R_adj = f_adj* ds(2)  + 2.0*f_adj*ds(3)    
    P,b_adj = [Function(V),assemble(R_adj)]    
    for bc in bcd: bc.apply(A,b_adj)    
    solver = LUSolver(A)   
    solver.solve(P.vector(), b_adj)           
    return P   
#-----------------------------------------------------------------------    
def _shape_der(Vvec, u_vec, p_vec, eps_er, mu, lmbda, dx, solver, Lag):   
    xi = TestFunction(Vvec)  
    rv = 0.0 
    for k in range(0,len(u_vec)):
        u,p = [u_vec[k],p_vec[k]]     
        eu,Du,Dxi,ep,Dp = [sym(grad(u)),grad(u),grad(xi),sym(grad(p)),grad(p)]         
        S1 = 2*mu * (- inner( ep, Du*Dxi ) - inner( eu, Dp*Dxi ) + inner(eu,ep)*div(xi))\
         +lmbda * (-div(p)*inner( Du.T, Dxi ) - div(u)*inner( Dp.T, Dxi ) + div(u)*div(p)*div(xi) )
        rv += -assemble( eps_er*S1*dx(0) + S1*dx(1) + S1*dx(2)\
         + Lag*div(xi)* dx(1) + Lag*div(xi)* dx(2))
    th = Function(Vvec)                  
    solver.solve(th.vector(), rv)
    return th
#-----------------------------------------------------------------------        
def _hj(v,psi,lx,ly,Nx,Ny,beta): 
    for k in range(10):
        Dym = Ny*np.repeat(np.diff(psi,axis=0),[2]+[1]*(Ny-1),axis=0)/ly 
        Dyp = Ny*np.repeat(np.diff(psi,axis=0),[1]*(Ny-1)+[2],axis=0)/ly
        Dxm = Nx*np.repeat(np.diff(psi),[2]+[1]*(Nx-1),axis=1)/lx 
        Dxp = Nx*np.repeat(np.diff(psi),[1]*(Nx-1)+[2],axis=1)/lx          
        g = 0.5*( v[0]*(Dxp + Dxm) + v[1]*(Dyp + Dym)) \
          - 0.5*(np.abs(v[0])*(Dxp - Dxm) + np.abs(v[1])*(Dyp - Dym)) 
        maxv = np.max(abs(v[0]) + abs(v[1]))
        dt  = beta*lx / (Nx*maxv)
        psi = psi - dt*g
    return  psi         
#-----------------------------------------------------------------------         
def _reinit(lx,ly,Nx,Ny,psi):      
    Dxs = Nx*(np.repeat(np.diff(psi),[2]+[1]*(Nx-1),axis=1) \
          +np.repeat(np.diff(psi),[1]*(Nx-1)+[2],axis=1))/(2*lx) 
    Dys = Ny*(np.repeat(np.diff(psi,axis=0),[2]+[1]*(Ny-1),axis=0)\
          +np.repeat(np.diff(psi,axis=0),[1]*(Ny-1)+[2],axis=0))/(2*ly)                  
    signum = psi / np.power(psi**2 + ((lx/Nx)**2)*(Dxs**2+Dys**2),0.5)
    for k in range(0,2):      
        Dym = Ny*np.repeat(np.diff(psi,axis=0),[2]+[1]*(Ny-1),axis=0)/ly 
        Dyp = Ny*np.repeat(np.diff(psi,axis=0),[1]*(Ny-1)+[2],axis=0)/ly
        Dxm = Nx*np.repeat(np.diff(psi),[2]+[1]*(Nx-1),axis=1)/lx 
        Dxp = Nx*np.repeat(np.diff(psi),[1]*(Nx-1)+[2],axis=1)/lx               
        Gp = np.sqrt((np.maximum(Dxm,0))**2 + (np.minimum(Dxp,0))**2 \
           + (np.maximum(Dym,0))**2 + (np.minimum(Dyp,0))**2)
        Gm = np.sqrt((np.minimum(Dxm,0))**2 + (np.maximum(Dxp,0))**2 \
           + (np.minimum(Dym,0))**2 + (np.maximum(Dyp,0))**2)           
        g  = np.maximum(signum,0)*Gp + np.minimum(signum,0)*Gm
        psi  = psi - (0.5*lx/Nx)*(g - signum)       
    return psi   
#-----------------------------------------------------------------------
def _comp_lsf(px,py,phi,phi_mat,dofsV_max):               
    for dof in range(0,dofsV_max):              
        if np.rint(px[dof]) %2 == .0: 
            cx,cy = np.int_(np.rint([px[dof]/2,py[dof]/2]))                                            
            phi.vector()[dof] = phi_mat[cy,cx]
        else:
            cx,cy = np.int_(np.floor([px[dof]/2,py[dof]/2]))                      
            phi.vector()[dof] = 0.25*(phi_mat[cy,cx] + phi_mat[cy+1,cx]\
              + phi_mat[cy,cx+1] + phi_mat[cy+1,cx+1])    
    return phi            
# ----------------------------------------------------------------------
if __name__ == '__main__':
    _main()

